/*****************************************************************************
*  LPC15xx RC5 transmit demo program for MicroCore100 rev A board
*
*  1. Use SCT0 timer to send encoded RC5 frames @ SCT0_OUT0 (P0_11)
*  2. Use S2 (P2_5) key to send key code
*  3. Use MRT to generate timeout delay between RC5 frames
*****************************************************************************/

#include "board.h"
/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
#define RC5_DATA0()      Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 30, IOCON_MODE_PULLDOWN);	// input low (pull down)
#define RC5_DATA1()      Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 30, IOCON_MODE_PULLUP); 	// input high (pull up)

static uint8_t  state;                                 // driver state
static uint8_t  count;                                 // bit counter
static uint8_t  flag;                                  // frame send ready flag
static uint16_t frame;                                 // frame
/*****************************************************************************
 * Private functions
 ****************************************************************************/
void RC5_Init(void);
void RC5_Send(uint16_t fr);

/*****************************************************************************
 * Public functions
 ****************************************************************************/

void RC5_Send(uint16_t fr)
{
    frame = fr << 2;                                   // MSB first, 14 bits shift left 2
    state = 0;
    flag  = 0;                                         // clear ready flag
    count = 14;                                        // set bit count to 14

    Chip_SCT_EnableEventInt(LPC_SCT0 , SCT_EVT_4 );    // event 4 generates interrupt

    while (!flag) ;                                    // wait for ready flag
}

void SCT0_IRQHandler(void)
{
    if (count)                                         // send next bit if required
    {
        if (state)
        {
            if (frame & 0x8000)
            {
                RC5_DATA1();
            }
            else
            {
                RC5_DATA0();
            }
            state = 0;
            count --;
            frame <<= 1;
        }
        else
        {
            if (frame & 0x8000)
            {
                RC5_DATA0();
            }
            else
            {
                RC5_DATA1();
            }
            state = 1;
        }
    }
    else
    {
        RC5_DATA0();                                       	// inactive carrier pulse

        Chip_SCT_DisableEventInt(LPC_SCT0, (CHIP_SCT_EVENT_T) 0x00);			// disable interrupt
        flag = 1;                                          	// signal ready
    }
    Chip_SCT_SetEventFlag(LPC_SCT0, SCT_EVT_4);			// clear event 4 flag
}

void RC5_Init(void)
{
    RC5_DATA0();                                           					// SCT0_IN0 low

    Chip_SCT_Init(LPC_SCT0);												// enable the SCT0 clock

    Chip_SCT_Config(LPC_SCT0, SCT_CONFIG_32BIT_COUNTER |
    						  SCT_CONFIG_AUTOLIMIT_L   );					// unified timer and auto limit

    Chip_INMUX_SelectSCT0Src(0, SCT0_INMUX_PIO0_30);						// SCT0_IN0 at P0_30

    LPC_SCT0->OUTPUT      = (1 << 0);                      					// SCT0_OUT0 (IR output) is high

    Chip_SWM_MovablePortPinAssign(SWM_SCT0_OUT0_O , 0, 11); 				// P0_11 is SCT0_OUT0, ASSIGN7(15:8


/********************************************************************
*  SCT_L: low part configuration:
********************************************************************/

    Chip_SCT_SetMatchCountL(LPC_SCT0, SCT_MATCH_0, (SystemCoreClock / 48000)-1);		// 75% of 36 KHz
    Chip_SCT_SetMatchReloadL(LPC_SCT0, SCT_MATCH_0, (SystemCoreClock / 48000)-1);

    Chip_SCT_SetMatchCountL(LPC_SCT0, SCT_MATCH_1, (SystemCoreClock / 36000)-1);		// 36 KHz
    Chip_SCT_SetMatchReloadL(LPC_SCT0, SCT_MATCH_1, (SystemCoreClock / 36000)-1);


    Chip_SCT_EventState(LPC_SCT0, SCT_EVENT_0, 0x00000003);				// event 0 happens in all states
    Chip_SCT_EventControl(LPC_SCT0, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0	|	// MATCHSEL [3:0]   = related to match 0
                                                                          SCT_HEVENT_L		    |	// HEVENT   [4]     = use L state & match
                                                                          SCT_OUTSEL_L		    |	// OUTSEL   [5]     = select input from IOSEL
                                                                          SCT_IOSEL_RISEIN0	  |	// IOSEL    [9:6]   = select input 0
                                                                          SCT_IOCOND_HIGH	    |	// IOCOND   [11:10] = high level
                                                                          SCT_COMBMODE_AND	  |	// COMBMODE [13:12] = match AND IO condition
                                                                          SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                                          SCT_STATEEV_0		    |	// STATEV   [19:15] = no state change
                                                                          SCT_MATCHMEM		    |	// MATCHMEM [20]    = "equal" to match
                                                                          SCT_DIRECTION		    ));	// DIRECTION[22:21] = direction independent


    Chip_SCT_EventState(LPC_SCT0, SCT_EVENT_1, 0x00000003);				// event 1 happens in all states
    Chip_SCT_EventControl(LPC_SCT0, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH1	|	// MATCHSEL [3:0]   = related to match 1
                                                                          SCT_COMBMODE_MATCH	|	// COMBMODE [13:12] = match condition only
                                                                          SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                                          SCT_STATEEV_0		    ));	// STATEV   [19:15] = no state change


    Chip_SCT_ClearOutput(LPC_SCT0, SCT_OUTPUT_0, SCT_EVT_0);			// IR LED low  @ event 0
    Chip_SCT_SetOutput(LPC_SCT0, SCT_OUTPUT_0, SCT_EVT_1);				// IR LED high @ event 1

    LPC_SCT0->LIMIT_L        = (1 << 1);                   				// events 1 is used as counter limit

/********************************************************************
*  SCT_H: high part configuration:
********************************************************************/

    Chip_SCT_SetMatchCountH(LPC_SCT0, SCT_MATCH_0, (SystemCoreClock / 1125)-1);		// 36 KHz / 32 ticks = 1125
    Chip_SCT_SetMatchReloadH(LPC_SCT0, SCT_MATCH_0, (SystemCoreClock / 1125)-1);


    Chip_SCT_EventState(LPC_SCT0, SCT_EVENT_4, 0x00000003);				// event 4 only happens in all states
    Chip_SCT_EventControl(LPC_SCT0, SCT_EVENT_4, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0	|	// MATCHSEL [3:0]   = related to match_H 0
                                                                          SCT_HEVENT_H		    |	// HEVENT   [4]     = use H state & match
                                                                          SCT_COMBMODE_MATCH	|	// COMBMODE [13:12] = match condition only
                                                                          SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                                          SCT_STATEEV_0		    ));	// STATEV   [19:15] = no state change


    LPC_SCT0->LIMIT_H        = (1 << 4);                   					// event 4 is used as H counter limit

    NVIC_EnableIRQ(SCT0_IRQn);                             					// enable SCT interrupt

    Chip_SCT_ClearControl(LPC_SCT0,(SCT_CTRL_HALT_L |SCT_CTRL_HALT_H));      // unhalt both low and high counter

}


void MRT_IRQHandler(void)
{
	Chip_MRT_IntClear(LPC_MRT_CH0);                 			// acknowledge MRT irq
}

void MRT_Init(void)
{
	Chip_MRT_Init();									  		// enable MRT clock

	Chip_MRT_SetInterval(LPC_MRT_CH0, (SystemCoreClock / 8)|
									  (1U << 31)		   );	// 1/8 sec (125 msec) timeout


	Chip_MRT_SetMode(LPC_MRT_CH0, 	(MRT_MODE_T)  (  (0 << 1)		|
                                    MRT_MODE_REPEAT	));     		// enable + repeated mode
    NVIC_EnableIRQ(MRT_IRQn);                              		// enable Multi-Rate timer irq
}

int main(void)
{
  static uint8_t  trg = 1;
  static uint8_t  tgl = 1;
  static uint16_t frame;

  SystemCoreClockUpdate();
  Board_Init();

  Chip_Clock_SetSysClockDiv(6);



  Chip_Clock_EnablePeriphClock(	(CHIP_SYSCTL_CLOCK_T) ( SYSCTL_CLOCK_SWM 	  | 			// enable SWM clock
                                                        SYSCTL_CLOCK_GPIO2 	|			// enable GPIO port 2 clock
                                                        SYSCTL_CLOCK_MUX 	  |			// enable PMUX clock
                                                        SYSCTL_CLOCK_IOCON	));		    //enable IOCON clock


  MRT_Init();
  RC5_Init();

    while (1)
    {
        frame = (1   << 13) |                              			// set start (S) bit
                (1   << 12) |                              			// set field (F) bit
                (tgl << 11) |                             	 		// clr control / toggle (C) bit
                (20  << 6 ) |                              			// system  = 20
                (9   << 0 );                               			// command = 9

        if (Chip_GPIO_GetPinState(LPC_GPIO, 2, 5))                	// P2_5 high to low?
        {
            if (trg)
            {
                trg = 0;                                   			// trigger P2_5 low
            }
            RC5_Send(frame);
        }
        else if ((Chip_GPIO_GetPinState(LPC_GPIO,2, 5)) && !trg)    // P2_5 low to high?
        {
            trg = 1;                                       			// trigger P2_5 high
            tgl = !tgl;                                    			// toggle RC5 control bit
        }
        __WFI();
    }
}
